(function() {
    'use strict';

    const { registerPaymentMethod } = wc.wcBlocksRegistry;
    const { getSetting } = wc.wcSettings;
    const { createElement, useState, useEffect } = wp.element;
    const { __ } = wp.i18n;

    // Get settings from localized data
    const settings = getSetting('paytree_data', {});

    // IP Detection functionality
    const detectIPAddresses = () => {
        return new Promise((resolve) => {
            let ipv4 = null;
            let ipv6 = null;
            let completed = 0;

            const checkResult = () => {
                completed++;
                if (completed === 2) {
                    if (ipv6 && ipv6 !== 'Not available') {
                        resolve(ipv6);
                    } else if (ipv4 && ipv4 !== 'Not available') {
                        resolve(ipv4);
                    } else {
                        resolve('Not available');
                    }
                }
            };

            // Fetch IPv4
            fetch('https://api.ipify.org?format=json')
                .then((response) => response.json())
                .then((data) => {
                    ipv4 = data.ip;
                    checkResult();
                })
                .catch(() => {
                    ipv4 = 'Not available';
                    checkResult();
                });

            // Fetch IPv6
            fetch('https://api6.ipify.org?format=json')
                .then((response) => response.json())
                .then((data) => {
                    ipv6 = data.ip;
                    checkResult();
                })
                .catch(() => {
                    ipv6 = 'Not available';
                    checkResult();
                });
        });
    };

    // Get friendly name for a payment method
    const getFriendlyName = (method) => {
        // Use friendly names from settings if available
        if (settings.friendly_names && settings.friendly_names[method]) {
            return settings.friendly_names[method];
        }
        
        // Fallback to default friendly names
        const friendlyNames = {
            'card': 'Credit/Debit Card',
            'crypto': 'Cryptocurrency',
            'wire': 'Wire Transfer',
            'paypal': 'PayPal',
            'klarna': 'Klarna',
            'ideal': 'iDEAL',
            'bancontact': 'Bancontact'
        };
        return friendlyNames[method] || method.charAt(0).toUpperCase() + method.slice(1);
    };

    // Payment method content component
    const PaytreeContent = ({ eventRegistration, emitResponse }) => {
        const { onPaymentSetup } = eventRegistration;
        
        // Determine initial selected method
        let initialMethod = '';
        if (settings.default_method && settings.enabled_methods && settings.enabled_methods.includes(settings.default_method)) {
            initialMethod = settings.default_method;
        } else if (settings.enabled_methods && settings.enabled_methods.length > 0) {
            initialMethod = settings.enabled_methods[0];
        }
        
        const [selectedMethod, setSelectedMethod] = useState(initialMethod);
        const [userIP, setUserIP] = useState('');

        useEffect(() => {
            if (settings.enable_ip_detection === 'yes') {
                detectIPAddresses().then(setUserIP);
            }
        }, []);

        useEffect(() => {
            const unsubscribe = onPaymentSetup(() => {
                if (!selectedMethod) {
                    return {
                        type: emitResponse.responseTypes.ERROR,
                        message: __('Please select a payment method.', 'paytree'),
                    };
                }

                return {
                    type: emitResponse.responseTypes.SUCCESS,
                    meta: {
                        paymentMethodData: {
                            paytree_method: selectedMethod,
                            pt_user_ip: userIP,
                        },
                    },
                };
            });

            return unsubscribe;
        }, [selectedMethod, userIP, onPaymentSetup, emitResponse]);

        if (!settings.enabled_methods || settings.enabled_methods.length === 0) {
            return createElement('div', { className: 'paytree-payment-methods' },
                createElement('p', {}, __('No payment methods are currently available.', 'paytree'))
            );
        }

        // If only one method is enabled, show it as selected
        if (settings.enabled_methods.length === 1) {
            return createElement('div', { className: 'paytree-payment-methods' },
                createElement('p', {}, 
                    __('Payment method: ', 'paytree') + 
                    getFriendlyName(settings.enabled_methods[0])
                )
            );
        }

        // Multiple methods - show radio buttons with default selection
        return createElement('div', { className: 'paytree-payment-methods' },
            settings.enabled_methods.map((method) =>
                createElement('p', { key: method, className: 'form-row' },
                    createElement('input', {
                        type: 'radio',
                        id: `paytree_${method}`,
                        name: 'paytree_method',
                        value: method,
                        checked: selectedMethod === method,
                        onChange: (e) => setSelectedMethod(e.target.value),
                        required: true,
                    }),
                    createElement('label', {
                        htmlFor: `paytree_${method}`,
                        style: { marginLeft: '5px' }
                    }, getFriendlyName(method))
                )
            )
        );
    };

    // Payment method configuration
    const PaytreePaymentMethod = {
        name: 'paytree',
        label: createElement('span', {}, settings.title || __('Online Payments', 'paytree')),
        content: createElement(PaytreeContent),
        edit: createElement(PaytreeContent),
        canMakePayment: () => {
            return settings.enabled_methods && settings.enabled_methods.length > 0;
        },
        ariaLabel: settings.description || __('Pay securely using card, crypto, or wire transfer.', 'paytree'),
        supports: {
            features: settings.supports || ['products', 'refunds'],
        },
    };

    // Register the payment method
    registerPaymentMethod(PaytreePaymentMethod);
})();
