<?php
/**
 * Paytree Blocks Integration
 * 
 * Handles the integration of Paytree payment gateway with WooCommerce Blocks Checkout
 */

if (!defined('ABSPATH')) {
    exit();
}

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

final class Paytree_Blocks_Integration extends AbstractPaymentMethodType
{
    /**
     * Payment method name/id
     */
    protected $name = 'paytree';

    /**
     * Settings from the gateway
     */
    protected $settings = [];

    /**
     * Initialize the payment method type
     */
    public function initialize()
    {
        $this->settings = get_option('woocommerce_paytree_settings', []);
    }

    /**
     * Check if the payment method is active
     */
    public function is_active()
    {
        // First check if the gateway is enabled
        $enabled = $this->get_setting('enabled', 'no');
        
        if ($enabled !== 'yes') {
            return false;
        }
        
        // Check if at least one payment method is enabled
        $methods = ['card', 'crypto', 'wire', 'paypal', 'klarna', 'ideal', 'bancontact'];
        foreach ($methods as $method) {
            if ($this->get_setting("enable_{$method}_method") === 'yes') {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Get the payment method script handles
     */
    public function get_payment_method_script_handles()
    {
        wp_register_script(
            'paytree-blocks-payment-method',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/paytree-blocks-payment-method.js',
            [
                'wc-blocks-registry',
                'wc-settings',
                'wp-element',
                'wp-i18n',
                'wp-html-entities',
            ],
            '1.2.0',
            true
        );
        
        return ['paytree-blocks-payment-method'];
    }

    /**
     * Get the payment method data for the frontend
     */
    public function get_payment_method_data()
    {
        // Get enabled payment methods and their friendly names
        $enabled_methods = [];
        $friendly_names = [];
        $methods = ['card', 'crypto', 'wire', 'paypal', 'klarna', 'ideal', 'bancontact'];
        
        foreach ($methods as $method) {
            if ($this->get_setting("enable_{$method}_method") === 'yes') {
                $enabled_methods[] = $method;
                
                // Get custom rename or use default friendly name
                $rename = $this->get_setting("rename_{$method}", '');
                if (!empty($rename)) {
                    $friendly_names[$method] = $rename;
                } else {
                    // Default friendly names
                    $defaults = [
                        'card' => 'Credit/Debit Card',
                        'crypto' => 'Cryptocurrency',
                        'wire' => 'Wire Transfer',
                        'paypal' => 'PayPal',
                        'klarna' => 'Klarna',
                        'ideal' => 'iDEAL',
                        'bancontact' => 'Bancontact'
                    ];
                    $friendly_names[$method] = isset($defaults[$method]) ? $defaults[$method] : ucfirst($method);
                }
            }
        }
        
        // Get default method
        $default_method = $this->get_setting('default_method', 'card');
        // If default method is not enabled, use first enabled method
        if (!in_array($default_method, $enabled_methods) && !empty($enabled_methods)) {
            $default_method = $enabled_methods[0];
        }

        $data = [
            'title' => $this->get_setting('title', 'Online Payments'),
            'description' => $this->get_setting('description', 'Pay securely using card, crypto, or wire transfer.'),
            'icon' => plugin_dir_url(dirname(__FILE__)) . 'assets/img/paytree-icon.png',
            'enabled_methods' => $enabled_methods,
            'friendly_names' => $friendly_names,
            'default_method' => $default_method,
            'enable_ip_detection' => $this->get_setting('enable_ip_detection', 'yes'),
            'force_route' => $this->get_setting('force_route', ''),
            'supports' => $this->get_supported_features(),
        ];
        
        error_log('Paytree Blocks Integration: get_payment_method_data() returning: ' . json_encode($data));
        
        return $data;
    }

    /**
     * Get supported features
     */
    public function get_supported_features()
    {
        return [
            'products',
            'refunds',
        ];
    }

    /**
     * Get a setting value
     */
    public function get_setting($key, $default = '')
    {
        return isset($this->settings[$key]) ? $this->settings[$key] : $default;
    }
}
