<?php

if (!defined("ABSPATH")) {
    exit();
}

class Paytree_WooCommerce_Gateway extends WC_Payment_Gateway
{
    public $api_token;
    public $api_endpoint;
    public $payment_methods;

    // Declare properties to prevent PHP 8.2+ deprecation warnings
    public $logger;
    public $methods;
    public $enable_ip_detection;
    public $default_method;
    public $enable_card_method;
    public $enable_crypto_method;
    public $enable_wire_method;
    public $enable_paypal_method;
    public $enable_klarna_method;
    public $enable_ideal_method;
    public $enable_bancontact_method;
    public $enable_log_request_body;
    public $enable_log_request_resp;

    // Rename properties
    public $rename_card;
    public $rename_crypto;
    public $rename_wire;
    public $rename_paypal;
    public $rename_klarna;
    public $rename_ideal;
    public $rename_bancontact;

    // Force route(s)
    public $force_route_card;
    public $force_route_crypto;
    public $force_route_wire;
    public $force_route_paypal;
    public $force_route_klarna;
    public $force_route_ideal;
    public $force_route_bancontact;

    public function __construct()
    {
        if (is_checkout() && !is_wc_endpoint_url("order-pay")) {
            $this->icon = "";
        } else {
            $this->icon =
                plugin_dir_url(dirname(__FILE__)) .
                "assets/img/paytree-icon.png";
        }

        $this->id = "paytree";
        $this->has_fields = true;
        $this->method_title = "Paytree";
        $this->method_description =
            "Accept multiple payment methods and orchestrate payment flow between different gateways";

        $this->init_form_fields();
        $this->init_settings();

        $this->logger = wc_get_logger();

        $this->methods = [
            "card",
            "crypto",
            "wire",
            "paypal",
            "klarna",
            "ideal",
            "bancontact",
        ];

        $this->title = $this->get_option("title");
        $this->description = $this->get_option("description");
        $this->enabled = $this->get_option("enabled");
        $this->api_token = $this->get_option("api_token");
        $this->api_endpoint = $this->get_option("api_endpoint");
        $this->enable_ip_detection = $this->get_option("enable_ip_detection");
        $this->force_route = $this->get_option("force_route");
        $this->default_method = $this->get_option("default_method", "card");

        foreach ($this->methods as $method) {
            // Label(s)
            $this->{"enable_{$method}_method"} = $this->get_option(
                "enable_{$method}_method",
            );
            $this->{"rename_{$method}"} = $this->get_option(
                "rename_{$method}",
                "",
            );

            // Force route(s)
            $this->{"force_route_{$method}"} = $this->get_option(
                "force_route_{$method}",
                "",
            );
        }

        $this->enable_log_request_body = $this->get_option("log_request_body");
        $this->enable_log_request_resp = $this->get_option("log_request_resp");

        // Add actions for saving admin settings and handling payment
        add_action("woocommerce_update_options_payment_gateways_" . $this->id, [
            $this,
            "process_admin_options",
        ]);
    }

    public function init_form_fields()
    {
        $this->form_fields = include dirname(__FILE__) . "/form-fields.php";
    }

    public function process_admin_options()
    {
        parent::process_admin_options();
    }

    private function get_enabled_methods()
    {
        $enabled = [];
        foreach ($this->methods as $method) {
            if ($this->{"enable_{$method}_method"} === "yes") {
                $enabled[] = $method;
            }
        }
        return $enabled;
    }

    private function get_friendly_name($method)
    {
        // First check if there's a custom rename for this method
        $rename_option = $this->{"rename_{$method}"};
        if (!empty($rename_option)) {
            return $rename_option;
        }

        // Otherwise use default friendly names
        $friendly_names = [
            "card" => "Credit/Debit Card",
            "crypto" => "Cryptocurrency",
            "wire" => "Wire Transfer",
            "paypal" => "PayPal",
            "klarna" => "Klarna",
            "ideal" => "iDEAL",
            "bancontact" => "Bancontact",
        ];

        return isset($friendly_names[$method])
            ? $friendly_names[$method]
            : ucfirst($method);
    }

    public function payment_fields()
    {
        if ($this->description) {
            echo wpautop(wp_kses_post($this->description));
        }

        // IP detection field handled globally by WooCommerce hooks for 100% reliability

        $enabled_methods = $this->get_enabled_methods();

        if (!empty($enabled_methods)) {
            echo '<div class="paytree-payment-methods">';

            // Determine which method to pre-select
            $preselect = "";

            // Check if default_method is enabled
            if (in_array($this->default_method, $enabled_methods, true)) {
                $preselect = $this->default_method;
            } else {
                // Otherwise use first enabled method
                $preselect = $enabled_methods[0];
            }

            // Preserve POSTed selection on validation errors
            if (isset($_POST["paytree_method"])) {
                $preselect = wc_clean(wp_unslash($_POST["paytree_method"]));
            }

            foreach ($enabled_methods as $method) {
                $checked = $preselect === $method ? 'checked="checked"' : "";

                echo '<p class="form-row">
                        <input type="radio" id="paytree_' .
                    $method .
                    '"
                               name="paytree_method"
                               value="' .
                    $method .
                    '"
                               ' .
                    $checked .
                    '
                               required />
                        <label for="paytree_' .
                    $method .
                    '">' .
                    esc_html($this->get_friendly_name($method)) .
                    '</label>
                      </p>';
            }
            echo "</div>";
        }
    }

    private function generate_uuid4()
    {
        $data = random_bytes(16);
        $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
        $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);

        return vsprintf("%s%s-%s-%s-%s-%s%s%s", str_split(bin2hex($data), 4));
    }

    private function get_client_ip()
    {
        if (isset($_SERVER["HTTP_CF_CONNECTING_IP"])) {
            return $_SERVER["HTTP_CF_CONNECTING_IP"];
        }

        if (isset($_SERVER["HTTP_X_FORWARDED_FOR"])) {
            $ip_list = explode(",", $_SERVER["HTTP_X_FORWARDED_FOR"]);
            return trim($ip_list[0]);
        }

        if (isset($_SERVER["REMOTE_ADDR"])) {
            return $_SERVER["REMOTE_ADDR"];
        }

        return "";
    }

    public function process_payment($order_id)
    {
        global $woocommerce;

        $order = wc_get_order($order_id);
        if (!$order) {
            wc_add_notice(__("Order not found.", "paytree"), "error");
            return [
                "result" => "failure",
                "messages" => __("Order not found.", "paytree"),
            ];
        }

        $total = $order->get_total();
        $currency = get_woocommerce_currency_symbol();

        // Handle both classic checkout and Blocks checkout
        $selected_method = "";

        // Debug logging
        $this->logger->info("Processing payment - POST data:", [
            "source" => "woo-paytree",
            "post_keys" => array_keys($_POST),
            "has_paytree_method" => isset($_POST["paytree_method"]),
            "has_payment_method_data" => isset($_POST["payment_method_data"]),
        ]);

        if (isset($_POST["paytree_method"])) {
            $selected_method = sanitize_text_field($_POST["paytree_method"]);
        } elseif (
            isset($_POST["payment_method_data"]) &&
            is_array($_POST["payment_method_data"])
        ) {
            if (isset($_POST["payment_method_data"]["paytree_method"])) {
                $selected_method = sanitize_text_field(
                    $_POST["payment_method_data"]["paytree_method"],
                );
            }
        } elseif (
            isset($_POST["extensions"]) &&
            is_array($_POST["extensions"])
        ) {
            // WooCommerce Blocks might pass data through extensions
            if (isset($_POST["extensions"]["paytree"]["paytree_method"])) {
                $selected_method = sanitize_text_field(
                    $_POST["extensions"]["paytree"]["paytree_method"],
                );
            }
        }

        // If still no method selected, use the first enabled method as default
        if (empty($selected_method)) {
            $enabled_methods = $this->get_enabled_methods();

            if (!empty($enabled_methods)) {
                $selected_method = $enabled_methods[0]; // Use first enabled method as default
                $this->logger->info(
                    "Auto-selected default method: " . $selected_method,
                    ["source" => "woo-paytree"],
                );
            }
        }

        if (empty($selected_method)) {
            wc_add_notice(
                __("Please select a payment method.", "paytree"),
                "error",
            );
            $this->logger->error("No payment method selected", [
                "source" => "woo-paytree",
                "post_data" => $_POST,
            ]);
            return [
                "result" => "failure",
                "messages" => __("Please select a payment method.", "paytree"),
            ];
        }

        $user_ip = $this->get_client_ip();
        $transaction_ref = $this->generate_uuid4();

        if ($this->get_option("enable_ip_detection") == "yes") {
            if (isset($_POST["pt_user_ip"])) {
                $user_ip = sanitize_text_field($_POST["pt_user_ip"]);
            } elseif (
                isset($_POST["payment_method_data"]) &&
                is_array($_POST["payment_method_data"]) &&
                isset($_POST["payment_method_data"]["pt_user_ip"])
            ) {
                $user_ip = sanitize_text_field(
                    $_POST["payment_method_data"]["pt_user_ip"],
                );
            } elseif (
                isset($_POST["extensions"]) &&
                is_array($_POST["extensions"]) &&
                isset($_POST["extensions"]["paytree"]["pt_user_ip"])
            ) {
                $user_ip = sanitize_text_field(
                    $_POST["extensions"]["paytree"]["pt_user_ip"],
                );
            }
        }

        // Step 1: Create Payment Intent with Paytree API
        $response = $this->create_payment_intent(
            $order_id,
            $transaction_ref,
            $selected_method,
            $user_ip,
        );

        if ($response && isset($response->payment_link)) {
            // Step 2: Store correlation id
            $order = wc_get_order($order_id);
            $order->update_meta_data(
                "_paytree_transaction_ref",
                $transaction_ref,
            );
            $order->save();

            // Step 3: Redirect customer to payment page
            return [
                "result" => "success",
                "redirect" => $response->payment_link,
            ];
        } else {
            // Handle errors and show message to customer
            if (
                is_object($response) &&
                property_exists($response, "message") &&
                isset($response->message)
            ) {
                wc_add_notice("Payment error: " . $response->message, "error");
            } else {
                wc_add_notice(
                    "Payment gateway returned an error. Please try again or contact us if the issue persists.",
                    "error",
                );
            }

            // Log errors
            $this->logger->error("Failed to generate payment intent", [
                "source" => "woo-paytree",
                "backtrace" => false,
                "order_id" => $order->get_order_number(),
                "order_key" => $order->get_order_key(),
                "response" => $response,
            ]);

            return [
                "result" => "failure",
                "messages" => wc_get_notices("error"),
            ];
        }
    }

    private function create_payment_intent(
        $order_id,
        $transaction_ref,
        $payment_method,
        $user_ip,
    ) {
        $api_token = $this->api_token;
        $api_endpoint = $this->api_endpoint;
        $order = wc_get_order($order_id);

        // Prepare data to send to Paytree API
        $data = [
            "transaction_ref" => $transaction_ref,
            "client_ref" => $order->get_order_key(),
            "amount" => number_format($order->get_total(), 2, ".", ""), // Convert amount to the required format
            "amount_currency" => $order->get_currency(),
            "method" => $payment_method,
            "customer" => [
                "first_name" => $order->get_billing_first_name(),
                "last_name" => $order->get_billing_last_name(),
                "email" => $order->get_billing_email(),
                "phone" => $order->get_billing_phone(),
            ],
            "address" => [
                "street" => $order->get_billing_address_1()
                    ? $order->get_billing_address_1()
                    : $order->get_shipping_address_1(),
                "city" => $order->get_billing_city()
                    ? $order->get_billing_city()
                    : $order->get_shipping_city(),
                "state" => $order->get_billing_state()
                    ? $order->get_billing_state()
                    : $order->get_shipping_state(),
                "zip" => $order->get_billing_postcode()
                    ? $order->get_billing_postcode()
                    : $order->get_shipping_postcode(),
                "country" => $order->get_billing_country()
                    ? $order->get_billing_country()
                    : $order->get_shipping_country(),
            ],
            "session" => [
                "ip_address" => $user_ip,
                "user_agent" => $_SERVER["HTTP_USER_AGENT"],
            ],
            "callback" => [
                "notification_url" => site_url(
                    "/wp-json/paytree/v1/webhook/?pi_id={payment_intent_id}&transaction_ref={transaction_id}",
                ),
                "return_url" => $order->get_checkout_order_received_url(),
                "cancel_url" => wc_get_cart_url(),
            ],
            "metadata" => [
                "woo_host" => $_SERVER["HTTP_HOST"],
                "woo_order_id" => $order->get_id(),
                "woo_order_key" => $order->get_order_key(),
            ],
        ];

        // Check if force route is enabled for the payment method
        $should_force_route = $this->get_option(
            "force_route_{$payment_method}",
        );

        if (!empty($should_force_route)) {
            $data["force_route"] = $should_force_route;
        }

        // Define the fields to check if empty
        $fields_to_check = [
            "customer" => ["phone"],
            "address" => ["street", "city", "state", "zip", "country"],
        ];

        // Loop through each item (like 'customer', 'address') and check the fields
        foreach ($fields_to_check as $item => $fields) {
            foreach ($fields as $field) {
                if (empty($data[$item][$field])) {
                    unset($data[$item][$field]);
                }
            }
        }

        $response = wp_remote_post($api_endpoint, [
            "method" => "POST",
            "headers" => [
                "Authorization" => "Token " . $api_token,
                "Content-Type" => "application/json",
            ],
            "body" => json_encode($data),
        ]);

        if ($this->enable_log_request_body === "yes") {
            $this->logger->debug("API Request", [
                "source" => "woo-paytree",
                "backtrace" => false,
                "endpoint" => $api_endpoint,
                "order_id" => $order->get_order_number(),
                "order_key" => $order->get_order_key(),
                "data" => $data,
            ]);
        }

        if ($this->enable_log_request_resp === "yes") {
            $this->logger->debug("API Response", [
                "source" => "woo-paytree",
                "backtrace" => false,
                "endpoint" => $api_endpoint,
                "order_id" => $order->get_order_number(),
                "order_key" => $order->get_order_key(),
                "response" => json_decode(wp_remote_retrieve_body($response)),
            ]);
        }

        if (is_wp_error($response)) {
            return false;
        }

        return json_decode(wp_remote_retrieve_body($response));
    }
}
