<?php
/**
 * Plugin Name: Paytree
 * Plugin URI: https://paytree.tech
 * Description: A WooCommerce payment gateway plugin that integrates with Paytree API.
 * Version:     1.2.2
 * Author:      Paytree
 * Author URI:  https://paytree.tech
 */

// Exit if accessed directly
if (!defined("ABSPATH")) {
    exit();
}

// Force Load WooCommerce Classes
if (!class_exists("WC_Payment_Gateway")) {
    include_once WP_PLUGIN_DIR .
        "/woocommerce/includes/abstracts/abstract-wc-settings-api.php";
    include_once WP_PLUGIN_DIR .
        "/woocommerce/includes/abstracts/abstract-wc-payment-gateway.php";
}

add_action("plugins_loaded", "paytree_init_gateway_class");
function paytree_init_gateway_class()
{
    if (!class_exists("WooCommerce")) {
        add_action("admin_notices", function () {
            echo '<div class="error"><p><strong>Paytree:</strong> WooCommerce must be installed and activated for Paytree to work properly.</p></div>';
        });
        return;
    }

    if (!class_exists("WC_Payment_Gateway")) {
        return; // WooCommerce is not available
    }
    require_once plugin_dir_path(__FILE__) .
        "includes/class-paytree-woocommerce.php";

    // Load Blocks integration if WooCommerce Blocks is available
    if (
        class_exists(
            "Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType",
        )
    ) {
        require_once plugin_dir_path(__FILE__) .
            "includes/class-paytree-blocks-integration.php";
    }
}

// Add Paytree Payment Gateway to WooCommerce
add_filter("woocommerce_payment_gateways", "add_paytree_gateway");

function add_paytree_gateway($gateways)
{
    $gateways[] = "Paytree_WooCommerce_Gateway";
    return $gateways;
}

// Register Blocks payment method integration
add_action("woocommerce_blocks_payment_method_type_registration", function (
    $payment_method_registry,
) {
    if (class_exists("Paytree_Blocks_Integration")) {
        $payment_method_registry->register(new Paytree_Blocks_Integration());
    }
});

// Add payment method data for Blocks frontend
add_action("woocommerce_init", "paytree_enqueue_payment_data_script");

function paytree_enqueue_payment_data_script()
{
    if (is_admin() || !function_exists("wc_get_payment_gateway_by_id")) {
        return;
    }

    $gateway = wc_get_payment_gateway_by_id("paytree");
    if (!$gateway) {
        return;
    }

    $settings = $gateway->settings;
    $enabled_methods = [];
    $methods = [
        "card",
        "crypto",
        "wire",
        "paypal",
        "klarna",
        "ideal",
        "bancontact",
    ];

    foreach ($methods as $method) {
        if (
            isset($settings["enable_{$method}_method"]) &&
            $settings["enable_{$method}_method"] === "yes"
        ) {
            $enabled_methods[] = $method;
        }
    }

    wp_enqueue_script(
        "paytree-blocks-payment-method", // Enqueue your script first
        plugin_dir_url(__FILE__) . "assets/js/paytree-blocks.js", // Adjust if needed
        ["jquery"],
        null,
        true,
    );

    wp_localize_script("paytree-blocks-payment-method", "paytree_data", [
        "title" => $settings["title"] ?? "Online Payments",
        "description" =>
            $settings["description"] ??
            "Pay securely using card, crypto, or wire transfer.",
        "icon" => plugin_dir_url(__FILE__) . "assets/img/paytree-icon.png",
        "enabled_methods" => $enabled_methods,
        "enable_ip_detection" => $settings["enable_ip_detection"] ?? "yes",
        "force_route" => $settings["force_route"] ?? "",
        "supports" => ["products", "refunds"],
    ]);
}

// Activation and deactivation hooks
register_activation_hook(__FILE__, "paytree_activate");
register_deactivation_hook(__FILE__, "paytree_deactivate");

function paytree_activate()
{
    // Actions to perform during plugin activation, such as adding default settings
}

function paytree_deactivate()
{
    // Actions to perform during plugin deactivation
}

// Add webhook handler
add_action("rest_api_init", function () {
    register_rest_route("paytree/v1", "/webhook/", [
        "methods" => "GET",
        "callback" => "handle_paytree_webhook",
        "permission_callback" => "__return_true",
    ]);
});

// Add custom order status to track chargebacks
add_action("init", "setup_disputed_order_status");
add_filter("wc_order_statuses", "add_disputed_to_order_statuses");

function setup_disputed_order_status()
{
    register_post_status("wc-disputed", [
        "label" => "Disputed",
        "public" => true,
        "show_in_admin_status_list" => true,
        "show_in_admin_all_list" => true,
        "exclude_from_search" => false,
        "label_count" => _n_noop(
            'Disputed <span class="count">(%s)</span>',
            'Disputed <span class="count">(%s)</span>',
        ),
    ]);
}

function add_disputed_to_order_statuses($order_statuses)
{
    $new_order_statuses = [];
    foreach ($order_statuses as $key => $status) {
        $new_order_statuses[$key] = $status;
        if ("wc-processing" === $key) {
            $new_order_statuses["wc-disputed"] = "Disputed";
        }
    }
    return $new_order_statuses;
}

// Hook to enqueue the custom JavaScript on WooCommerce checkout page
function wc_ip_detection_enqueue_scripts()
{
    if (is_checkout()) {
        wp_enqueue_script(
            "wc-ip-detection-js",
            plugin_dir_url(__FILE__) . "assets/js/ip-detection.js",
            ["jquery"],
            null,
            true,
        );
    }
}
add_action("wp_enqueue_scripts", "wc_ip_detection_enqueue_scripts");

// Hook to add hidden field in WooCommerce checkout form - GLOBAL for 100% reliability
function wc_ip_detection_add_global_ip_field($checkout)
{
    if (!function_exists("WC")) {
        return;
    }

    $payment_gateways = WC()->payment_gateways();
    $gateway =
        $payment_gateways->get_available_payment_gateways()["paytree"] ?? null;

    if ($gateway && $gateway->get_option("enable_ip_detection") === "yes") {
        echo '<div id="wc-ip-detection-field">
            <input type="hidden" name="pt_user_ip" id="pt_user_ip" value="">
          </div>';
    }
}

add_action(
    "woocommerce_after_checkout_billing_form",
    "wc_ip_detection_add_global_ip_field",
);

// Hook to save IP address to order meta data
function wc_ip_detection_save_ip_to_order($order_id)
{
    if (isset($_POST["pt_user_ip"])) {
        update_post_meta(
            $order_id,
            "_user_ip",
            sanitize_text_field($_POST["pt_user_ip"]),
        );
    }
}
add_action(
    "woocommerce_checkout_update_order_meta",
    "wc_ip_detection_save_ip_to_order",
);

function handle_paytree_webhook(WP_REST_Request $request)
{
    $logger = wc_get_logger();
    $gateway_settings = get_option("woocommerce_paytree_settings");
    $api_base_endpoint = $gateway_settings["api_endpoint"];
    $api_token = $gateway_settings["api_token"];

    $payment_intent_id = $request->get_param("pi_id");
    $api_endpoint = $api_base_endpoint . $payment_intent_id . "/";
    $payment_status = "pending";

    if (isset($gateway_settings["api_endpoint"])) {
        $api_base_endpoint = $gateway_settings["api_endpoint"];
    } else {
        $logger->error("API endpoint not found in settings", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "endpoint" => $api_endpoint,
        ]);

        return new WP_REST_Response(
            ["error" => "PAYTREE_API_ENDPOINT_MISSING"],
            500,
        );
    }

    if (isset($gateway_settings["api_token"])) {
        $api_token = $gateway_settings["api_token"];
    } else {
        $logger->error("API token not found in settings", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "endpoint" => $api_endpoint,
        ]);

        return new WP_REST_Response(
            ["error" => "PAYTREE_API_TOKEN_MISSING"],
            500,
        );
    }

    $response = wp_remote_request($api_endpoint, [
        "method" => "GET",
        "headers" => [
            "Authorization" => "Token " . $api_token,
        ],
    ]);

    if (is_wp_error($response)) {
        $logger->error("Failed to fetch payment intent via API.", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "endpoint" => $api_endpoint,
            "response" => $response,
        ]);

        return new WP_REST_Response(
            ["error" => "PAYTREE_API_REQUEST_FAIL"],
            500,
        );
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (!isset($data["id"])) {
        $logger->error("Failed to parse payment intent data from API.", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "endpoint" => $api_endpoint,
            "response" => $response,
        ]);

        return new WP_REST_Response(["error" => "PAYTREE_API_PARSE_FAIL"], 500);
    }

    $transaction_ref = $data["transaction_ref"];
    $orders = wc_get_orders([
        "limit" => 1,
        "meta_key" => "_paytree_transaction_ref",
        "meta_value" => $transaction_ref,
        "return" => "ids",
    ]);

    $order_id = $orders ? $orders[0] : null;
    if ($order_id == null) {
        // Every time a user hits "Checkout" a new request is made to Paytree.
        // In most cases, subsequent requests will be received. These can be ignored.
        // If not ignored, it will update the status of an order incorrectly, a bug.
        return new WP_REST_Response(["error" => "WP_OK_SKIP"], 200);
    }

    $order = wc_get_order($order_id);
    if (!$order) {
        $logger->error("Failed to get order wc_get_order", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "wc_order_id" => $order_id,
        ]);

        return new WP_REST_Response(["error" => "WP_FAILED_WC_GET_ORDER"], 500);
    }

    // Set payment status to value in payload
    $payment_status = $data["payment"][0]["status"];
    $current_status = $order->get_status();

    // Update payment status in WooCommerce
    if ($payment_status === "success") {
        $order->payment_complete();
    } elseif ($payment_status === "pending") {
        $order->update_status("pending", "Payment pending via Paytree");
    } elseif ($payment_status === "partial") {
        $order->update_status("partial", "Payment partial via Paytree");
    } elseif ($payment_status === "cancelled") {
        $order->update_status("cancelled", "Payment cancelled via Paytree");
    } elseif ($payment_status === "expired") {
        $order->update_status("expired", "Payment expired via Paytree");
    } elseif ($payment_status === "authorized") {
        $order->update_status("authorized", "Payment authorized via Paytree");
    } elseif ($payment_status === "refunded") {
        $order->update_status("refunded", "Payment refunded via Paytree");
    } elseif ($payment_status === "partial_refunded") {
        $order->update_status(
            "partial_refunded",
            "Payment partial refunded via Paytree",
        );
    } elseif ($payment_status === "failed") {
        $order->update_status("failed", "Payment failed via Paytree");
    } elseif (
        $payment_status === "disputed" ||
        $payment_status === "disputed_won" ||
        $payment_status === "disputed_lost" ||
        $payment_status === "disputed_closed"
    ) {
        $order->update_status(
            "wc-disputed",
            "Payment disputed via Paytree (Chargeback)",
        );
    } else {
        $logger->error("Received invalid event from webhook", [
            "source" => "woo-paytree",
            "backtrace" => false,
            "payment_intent_id" => $payment_intent_id,
            "wc_order_id" => $order_id,
            "payload" => $data,
        ]);

        return new WP_REST_Response(
            ["error" => "WEBHOOK_FAILED_INVALID_EVENT"],
            400,
        );
    }

    return new WP_REST_Response("OK", 200);
}
